<?php

if (!defined('ABSPATH')) {
    exit;
}

require_once FINMO_PAYMENTS_PLUGIN_PATH . 'includes/finmo-webhooks.php';

/**
 * WooCommerce Payment Gateway class for Finmo.
 */
class FinmoPayments_Gateway extends WC_Payment_Gateway
{
    /**
     * Constructor for the payment gateway.
     */
    public function __construct()
    {
        $this->id                 = 'finmo_payments';
        $this->icon               = ''; // handled via custom HTML
        $this->has_fields         = true;
        $this->method_title       = 'Finmo Payments';

        $product_guides_url = 'https://docs.finmo.net/docs/woocommerce#/';
        $dashboard_url      = 'https://dashboard.finmo.net/preview/dashboard/settings/developer-settings/plugins';
        $get_started_url    = 'https://dashboard.finmo.net/signup';

        $this->method_description = sprintf(
            __('<p>Finmo for WooCommerce enables merchants to accept payments (mostly instant payment methods) in their customers\' preferred currency. 
            To get started, you can find your API access & secret keys in the <a href="%1$s" target="_blank" rel="noopener noreferrer">Developer Settings</a> on your Finmo Dashboard. You can find additional information in our 
            <a href="%2$s" target="_blank" rel="noopener noreferrer">product guides</a>. If you are new to Finmo, <a href="%3$s" target="_blank" rel="noopener noreferrer">click here to get started</a>.</p>', FINMO_PAYMENTS_PLUGIN_NAME),
            esc_url($dashboard_url),
            esc_url($product_guides_url),
            esc_url($get_started_url)
        );

        $this->supports = array('products', 'refunds');

        $this->form_fields = $this->finmo_config_form_fields();
        $this->init_settings();
        $this->finmo_assign_admin_settings();

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'finmo_update_setting_payment_fields'));

        // Webhooks
        add_action('woocommerce_api_finmo_checkout', 'finmo_checkout_webhook_update');
        add_action('woocommerce_api_finmo_refund', 'finmo_refund_webhook_update');
        // Frontend customizations
        add_filter('woocommerce_gateway_title', array($this, 'finmo_gateway_title_as_image'), 10, 2);
        add_filter('woocommerce_gateway_description', array($this, 'finmo_gateway_description_custom'), 10, 2);
        add_filter('woocommerce_order_get_payment_method_title', function ($title, $order) {
            if ($order->get_payment_method() === 'finmo_payments') {
                return 'Finmo Payments'; // plain text for order pages
            }
            return $title;
        }, 10, 2);

        $this->frontend_styles = new Finmo_Frontend_Styles($this->id);
        add_action('wp_enqueue_scripts', array($this->frontend_styles, 'enqueue_frontend_styles'));
    }


    private function build_checkout_args($order)
    {
        $items = [];
        foreach (WC()->cart->get_cart() as $cart_item) {
            $items[] = $cart_item['quantity'] . ' x ' . $cart_item['data']->get_title();
        }

        $description = get_bloginfo('name') . ' : ' . implode(', ', $items);

        $webhook_url = add_query_arg(
            ['wc-api' => 'finmo_checkout', 'order_id' => $order->get_id()],
            home_url('/')
        );

        $return_url = add_query_arg(
            ['wc-api' => 'finmo_return', 'order_id' => $order->get_id()],
            home_url('/')
        );

        return [
            'amount'      => floatval($order->get_total()),
            'currency'    => $order->get_currency(),
            'country'     => $order->get_billing_country(),
            'webhook_url' => $webhook_url,
            'order_id'    => strval($order->get_id()),
            'description' => $description,
            'success_url' => $return_url,
            'cancel_url'  => wc_get_checkout_url(),
            'return_url'  => $return_url,
        ];
    }

    /**
     * Build refund arguments for Finmo API.
     */
    private function build_refund_args($order, $amount, $checkout_id)
    {
        $webhook_url = add_query_arg(
            ['wc-api' => 'finmo_refund', 'order_id' => $order->get_id()],
            home_url('/')
        );

        return [
            "organization_reference_id" => $order->get_id(),
            "order_id"                  => $order->get_id(),
            "amount"                    => floatval($amount),
            "checkout_id"               => $checkout_id,
            "currency"                  => $order->get_currency(),
            "webhook_url"               => $webhook_url,
        ];
    }

    /**
     * Get environment keys.
     */
    private function get_environment_keys()
    {
        $api_environment = $this->get_option("api_environment");
        if ($api_environment === "Production") {
            finmo_log("Using Production environment keys", 'info');
            return [$this->get_option("prod_access_key"), $this->get_option("prod_secret_key"), "production"];
        }
        finmo_log("Using Sandbox environment keys", 'info');
        return [$this->get_option("sandbox_access_key"), $this->get_option("sandbox_secret_key"), "sandbox"];
    }

    /**
     * Handle API errors and log them.
     */
    private function handle_api_errors($order, $result, $context = '')
    {
        $payment_err_msg = '';
        if (!empty($result->error->message)) {
            foreach ($result->error->message as $error) {
                $payment_err_msg .= '<br>' . esc_html($error);
            }
        } elseif (!empty($result->error)) {
            foreach ($result->error as $error) {
                $payment_err_msg .= '<br>' . esc_html($error);
            }
        }

        if (!empty($payment_err_msg)) {
            $order->add_order_note("Finmo $context Failed: $payment_err_msg");
            finmo_log("Finmo $context Error for order {$order->get_id()}: $payment_err_msg", 'error');
            return wc_add_notice($payment_err_msg, 'error');
        }
    }

    /**
     * Define the payment gateway form fields.
     */
    public function finmo_config_form_fields()
    {

        return [
            'enabled' => [
                'title' => __('Enable/Disable', FINMO_PAYMENTS_PLUGIN_NAME),
                'label' => __('Enable Finmo Payments', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'checkbox',
                'default' => 'no',
            ],
            'api_environment' => [
                'title' => __('Environment', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'select',
                'options' => ['Production' => 'Production', 'Sandbox' => 'Sandbox'],
                'default' => 'Production',
                'description' => __('Use Sandbox for testing transactions without real money, and switch to Production when you’re ready to accept live payments.', FINMO_PAYMENTS_PLUGIN_NAME),
                'desc_tip' => true,
            ],
            'prod_access_key' => [
                'title' => __('Production Access Key', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'password',
                'description' => __('Key that identifies your store for live payments.', FINMO_PAYMENTS_PLUGIN_NAME),
                'desc_tip' => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            'prod_secret_key' => [
                'title' => __('Production Secret Key', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'password',
                'description' => __('Key that authorizes live payments. Keep this secure and never share.', FINMO_PAYMENTS_PLUGIN_NAME),
                'desc_tip' => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            'sandbox_access_key' => [
                'title' => __('Sandbox Access Key', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'password',
                'description' => __('Key that identifies your store for test payments in Sandbox.', FINMO_PAYMENTS_PLUGIN_NAME),
                'desc_tip' => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            'sandbox_secret_key' => [
                'title' => __('Sandbox Secret Key', FINMO_PAYMENTS_PLUGIN_NAME),
                'type' => 'password',
                'description' => __('Key that authorizes test payments in Sandbox. Safe for testing only', FINMO_PAYMENTS_PLUGIN_NAME),
                'desc_tip' => true,
                'custom_attributes' => ['required' => 'required'],
            ],
        ];
    }

    public function finmo_assign_admin_settings()
    {
        $this->title       = __('Finmo Payments', FINMO_PAYMENTS_PLUGIN_NAME);
        $this->enabled     = $this->get_option('enabled');
        $this->description = __('Supports instant payments and bank transfers | Powered by Finmo', FINMO_PAYMENTS_PLUGIN_NAME);
    }

    public function finmo_gateway_title_as_image($title, $payment_id)
{

    if ($payment_id !== $this->id) {
        return $title;
    }


    $uri = $_SERVER['REQUEST_URI'];
    $is_thankyou      = function_exists('is_order_received_page') && is_order_received_page();
    $is_view_order    = (strpos($uri, 'view-order') !== false);
    $is_order_received = (strpos($uri, 'order-received') !== false);

    if ($is_thankyou || $is_view_order || $is_order_received || !is_checkout()) {
        return 'Finmo Payments';
    }


    $logo_url = FINMO_PAYMENTS_PLUGIN_URL . 'assets/finmo-logo.png';
    return '<div class="finmo-payment-method-display">
                <div class="finmo-company-info">
				<span class="finmo-company-name">Finmo Payments</span>
                    <img src="' . esc_url($logo_url) . '" alt="Finmo" class="finmo-logo" />
                </div>
            </div>';
}
    public function finmo_gateway_description_custom($description, $payment_id)
    {
        if ($payment_id !== $this->id) return $description;
        if (function_exists('is_order_received_page') && is_order_received_page()) return '';
        if (isset($_GET['order-received']) || strpos($_SERVER['REQUEST_URI'], 'order-received') !== false) return '';
        return '<p class="payment_box payment_method_' . esc_attr($this->id) . '">' .
            esc_html__('Supports instant payments and bank transfers | Powered by Finmo', FINMO_PAYMENTS_PLUGIN_NAME) .
            '</p>';
    }

    public function finmo_update_setting_payment_fields()
    {
        $this->init_settings();
        $post_data = $this->get_post_data();
        $error_fields = [];

        $api_key_req = [
            "plugin_type" => "woocommerce",
            "production" => [
                "access_key" => $post_data['woocommerce_finmo_payments_prod_access_key'],
                "secret_key" => $post_data['woocommerce_finmo_payments_prod_secret_key'],
            ],
            "sandbox" => [
                "access_key" => $post_data['woocommerce_finmo_payments_sandbox_access_key'],
                "secret_key" => $post_data['woocommerce_finmo_payments_sandbox_secret_key'],
            ],
        ];

        $api_validation = finmo_validate_api_key($api_key_req);
        if ($api_validation->success === false) {
            finmo_log("API key validation failed: {$api_validation->status_code}", 'error');
            WC_Admin_Settings::add_error(__('Invalid API Keys or Server Error.', FINMO_PAYMENTS_PLUGIN_NAME));
            $error_fields = ['prod_access_key', 'prod_secret_key'];
        }

        foreach ($this->get_form_fields() as $key => $field) {
            if (!in_array($key, $error_fields)) {
                $this->settings[$key] = $this->get_field_value($key, $field, $post_data);
            }
        }

        return update_option($this->get_option_key(), $this->settings);
    }

    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);
        if (!$order) return wc_add_notice(__('Order not found.', 'woocommerce'), 'error');

        $checkout_args = $this->build_checkout_args($order);
        list($access_key, $secret_key, $current_env) = $this->get_environment_keys();

        try {
            $result = finmo_create_checkout_request($access_key, $secret_key, $current_env, $checkout_args, $order_id);
            finmo_log("Checkout created for order {$order_id}", 'info');
        } catch (Exception $e) {
            $order->add_order_note('Finmo Checkout Failed: ' . esc_html($e->getMessage()));
            finmo_log("Checkout Exception: " . $e->getMessage(), 'error');
            return wc_add_notice(__('Payment could not be processed. Please try again.', 'woocommerce'), 'error');
        }

        if ($result->success === false) return $this->handle_api_errors($order, $result, 'Checkout');

        $order->add_order_note('Finmo Checkout created - ' . $result->data->checkout_id);
        update_post_meta($order_id, 'finmo_checkout_id', $result->data->checkout_id);
        update_post_meta($order_id, 'finmo_checkout_url', $result->data->checkout_url);

        return ['result' => 'success', 'redirect' => $result->data->checkout_url];
    }

    public function process_refund($order_id, $amount = null, $reason = 'refund')
    {
        $order = wc_get_order($order_id);
        if (!$order) return new WP_Error('error', __('Refund failed: Order Not Found', 'woocommerce'));
        $is_refundable = get_post_meta($order_id, '_finmo_is_refundable', true) === 'yes';
        $is_partially_refundable = get_post_meta($order_id, '_finmo_is_partially_refundable', true) === 'yes';
        finmo_log($is_refundable,$is_partially_refundable);
        if (!$is_refundable) {
            return new WP_Error('error', __('Refund failed: This order is not refundable', 'woocommerce'));
        }

        if (!$is_partially_refundable && $amount !== null) {
            $order_total = $order->get_total();
            if ($amount < $order_total) {
                return new WP_Error('error', __('Partial refund not allowed for this order', 'woocommerce'));
            }
        }

        if ($order->get_payment_method() == 'finmo_payments') {
            $checkout_id = get_post_meta($order_id, 'finmo_checkout_id', true);
            $refund_arg = $this->build_refund_args($order, $amount, $checkout_id);
            list($access_key, $secret_key, $current_env) = $this->get_environment_keys();

            try {
                $result = finmo_create_refund_request($access_key, $secret_key, $current_env, $refund_arg, $order_id);
                finmo_log("Refund request for order {$order_id}, amount {$amount}", 'info');
            } catch (Exception $e) {
                $order->add_order_note('Finmo Refund Failed: ' . esc_html($e->getMessage()));
                finmo_log("Refund Exception: " . $e->getMessage(), 'error');
                return new WP_Error('error', __('Refund failed. Please try again later.', 'woocommerce'));
            }

            if ($result->success === false) return $this->handle_api_errors($order, $result, 'Refund');

            update_post_meta($order_id, 'finmo_refund_id', $result->data->refund_id);
            $order->add_order_note('Finmo Refund requested.');
            return true;
        }
        return false;
    }
    
}